<?php
/*--------------------------------------------------------------
   UpdateProductOptionsSortOrderAction.php 2021-05-05
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2021 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
 -------------------------------------------------------------*/

declare(strict_types=1);

namespace Gambio\Api\Modules\ProductOption\App\Actions;

use Exception;
use Gambio\Admin\Modules\ProductOption\Services\ProductOptionReadService as ProductOptionReadServiceInterface;
use Gambio\Admin\Modules\ProductOption\Services\ProductOptionWriteService as ProductOptionWriteServiceInterface;
use Gambio\Api\Modules\ProductOption\App\ProductOptionApiRequestValidator;
use Gambio\Core\Application\Http\Request;
use Gambio\Core\Application\Http\Response;
use InvalidArgumentException;

/**
 * Class UpdateProductOptionsSortOrderAction
 * @package Gambio\Api\Modules\ProductOption\App\Actions
 */
class UpdateProductOptionsSortOrderAction
{
    /**
     * @var ProductOptionApiRequestValidator
     */
    private $validator;
    
    /**
     * @var ProductOptionReadServiceInterface
     */
    private $readService;
    
    /**
     * @var ProductOptionWriteServiceInterface
     */
    private $writeService;
    
    
    /**
     * UpdateProductOptionsSortOrderAction constructor.
     *
     * @param ProductOptionApiRequestValidator   $validator
     * @param ProductOptionReadServiceInterface  $readService
     * @param ProductOptionWriteServiceInterface $writeService
     */
    public function __construct(
        ProductOptionApiRequestValidator $validator,
        ProductOptionReadServiceInterface $readService,
        ProductOptionWriteServiceInterface $writeService
    ) {
        $this->validator    = $validator;
        $this->readService  = $readService;
        $this->writeService = $writeService;
    }
    
    
    /**
     * @param Request  $request
     * @param Response $response
     * @param array    $args
     *
     * @return Response
     */
    public function __invoke(Request $request, Response $response, array $args): Response
    {
        if (($productId = (int)$request->getAttribute('productId')) === 0) {
            
            return $response->withStatus(400)->withJson(['errors' => ['product id can\'t be 0']]);
        }
        
        $errors = $this->validator->validateUpdateSortOrderRequest($request->getParsedBody());
        
        if (empty($errors) === false) {
            
            return $response->withStatus(400)->withJson(['errors' => $errors]);
        }
        
        $productOptions = [];
        
        try {
            
            foreach ($request->getParsedBody() as $documentData) {
                
                $productOptions[] = $productOption = $this->readService->getProductOptionById((int)$documentData['id']);
                
                if ($productOption->productId() !== $productId) {
                    
                    throw new InvalidArgumentException(sprintf('Product option with id "%s" belongs to the product with the id "%s"',
                                                               $productOption->id(),
                                                               $productOption->productId()));
                }
                
                $productOption->changeSortOrder((int)$documentData['sortOrder']);
            }
            
            $this->writeService->storeProductOptions(...$productOptions);

            return $response->withStatus(204);
        } catch (Exception $exception) {
            
            return $response->withStatus(422)->withJson(['errors' => [[$exception->getMessage()]]]);
        }
    }
}